// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef SERVICES_TRACING_PUBLIC_CPP_TRACE_STARTUP_CONFIG_H_
#define SERVICES_TRACING_PUBLIC_CPP_TRACE_STARTUP_CONFIG_H_

#include "base/component_export.h"
#include "base/files/file_path.h"
#include "base/trace_event/trace_config.h"
#include "build/build_config.h"
#include "third_party/perfetto/include/perfetto/tracing/core/trace_config.h"

namespace base {
template <typename Type>
class NoDestructor;
}  // namespace base

namespace content {
class CommandlineStartupTracingTest;
class BackgroundStartupTracingTest;
}

namespace tracing {
class TraceStartupConfigTest;
class TraceStartupSharedMemoryTest;

// TraceStartupConfig is a singleton that contains the configurations of startup
// tracing. One can use --trace-startup flag or, for more complicated
// configurations, create a trace config file and use it to configure startup
// tracing.
//
// The trace config file should be JSON formated. One example is:
//   {
//     "trace_config": {
//        "record_mode": "record-until-full",
//        "included_categories": ["cc", "skia"]
//      },
//      "startup_duration": 5,
//      "result_file": "chrometrace.log"
//   }
//
// trace_config:     The configuration of tracing. Please see the details in
//                   base/trace_event/trace_config.h.
//
// startup_duration: The duration for startup tracing in terms of seconds.
//                   Tracing will stop automatically after the duration. If this
//                   value is not specified, the duration is 0 and one needs
//                   to stop tracing by other ways, e.g., by DevTools, or get
//                   the result file after shutting the browser down.
//
// result_file:      The file that contains the trace log. The default result
//                   file path is chrometrace.log, except on Android, where the
//                   default path is generated by tracing controller. Chrome
//                   will dump the trace log to this file
//                   1) after startup_duration if it is specified;
//                   2) or after browser shutdown if startup duration is 0.
//                   One can also stop tracing and get the result by other ways,
//                   e.g., by DevTools. In that case, the trace log will not be
//                   saved to this file.
//
// result_directory: A directory to save traces to. The saved traces will be
//                   named chrometrace.log, prefixed with the Unix timestamp of
//                   Chrome's startup time. This is meant for cases where Chrome
//                   is started multiple times from one command line, such as
//                   during tests. This option is ignored if result_file is also
//                   specified.
//
// The trace config file can be specified by the --trace-config-file flag on
// most platforms except on Android, e.g., --trace-config-file=path/to/file/.
// This flag should not be used with --trace-startup; otherwise,
// --trace-config-file flag will be ignored. If the --trace-config-file flag is
// used without the file path, Chrome will do startup tracing with 5 seconds'
// duration.
//
// On Android, Chrome does not read the --trace-config-file flag, because not
// all Chrome based browsers read customized flag, e.g., Android WebView. Chrome
// on Android reads from a fixed file location:
//   /data/local/chrome-trace-config.json
// If this file exists, Chrome will start tracing according to the configuration
// specified in the file, otherwise, Chrome will not start tracing.
//
// In order for Chrome to read this file on Android M+, SE Linux mode must be
// set to 0. Run "adb shell setenforce 0". Note that on Android, the output path
// is ignored, and the result is emitted to a location generated by
// TracingControllerAndroid::GenerateTracingFilePath.
class COMPONENT_EXPORT(TRACING_CPP) TraceStartupConfig {
 public:
  enum class SessionOwner {
    kTracingController,
    kDevToolsTracingHandler,
    // The background tracing config set in application preferences on the
    // previous session, for current session.
    kBackgroundTracing,
    // We expect the System tracing to take over.
    kSystemTracing
  };

  enum class OutputFormat {
    kLegacyJSON,
    kProto,
  };

  static TraceStartupConfig& GetInstance();

  ~TraceStartupConfig();

  TraceStartupConfig(const TraceStartupConfig&) = delete;
  TraceStartupConfig& operator=(const TraceStartupConfig&) = delete;

  // Default minimum startup trace config with enough events to debug issues.
  static perfetto::TraceConfig GetDefaultBackgroundStartupConfig();

  // IsEnabled() returns true if
  // - valid trace config file or trace startup flags are specified,
  // - the specified startup duration is zero or we are not passed the positive
  //   startup duration, and
  // - startup tracing is not stopped by other means, e.g. via DevTools
  //   protocol.
  bool IsEnabled() const;

  // SetDisabled() is used by the tracing controller to indicate that startup
  // tracing is finished.
  void SetDisabled();

  perfetto::TraceConfig GetPerfettoConfig() const;

  // Returns the name of the file to write the trace result into.
  base::FilePath GetResultFile() const;

  // Set the background tracing config in preferences for the next session.
  void SetBackgroundStartupTracingEnabled(bool enabled);

  SessionOwner GetSessionOwner() const;

  OutputFormat GetOutputFormat() const;

  // Called by a potential session owner to determine if it should take
  // ownership of the startup tracing session and begin tracing. Returns |true|
  // if the passed |owner| should adopt the session.
  bool AttemptAdoptBySessionOwner(SessionOwner owner);

 private:
  // This allows constructor and destructor to be private and usable only
  // by the NoDestructor class.
  friend class base::NoDestructor<TraceStartupConfig>;
  friend class content::CommandlineStartupTracingTest;
  friend class content::BackgroundStartupTracingTest;
  friend class ::tracing::TraceStartupConfigTest;
  friend class ::tracing::TraceStartupSharedMemoryTest;

  constexpr static int kDefaultStartupDurationInSeconds = 5;

  TraceStartupConfig();

  bool EnableFromCommandLine();
  bool EnableFromConfigFile();
  bool EnableFromConfigHandle();
  bool EnableFromBackgroundTracing();

  bool ParseTraceConfigFileContent(const std::string& content);

  bool is_enabled_ = false;
  perfetto::TraceConfig perfetto_config_;
  base::FilePath result_file_;
  SessionOwner session_owner_ = SessionOwner::kTracingController;
  bool session_adopted_ = false;
  OutputFormat output_format_ = OutputFormat::kProto;
};

}  // namespace tracing

#endif  // SERVICES_TRACING_PUBLIC_CPP_TRACE_STARTUP_CONFIG_H_
