
=pod

=head1 NAME

acl::Env - Mediate read access to the environment used by the Intel(R) FPGA SDK for OpenCL(TM).

=head1 COPYRIGHT

# (c) 1992-2024 Intel Corporation.                                              
# Intel, the Intel logo, Intel, MegaCore, NIOS II, Quartus and TalkBack         
# words and logos are trademarks of Intel Corporation or its                    
# subsidiaries in the U.S. and/or other countries. Other marks and              
# brands may be claimed as the property of others.                              
# See Trademarks on intel.com for full list of Intel trademarks or the          
# Trademarks & Brands Names Database (if Intel)                                 
# or See www.Intel.com/legal (if Altera)                                        
# Your use of Intel Corporation's design tools, logic functions and             
# other software and tools, and its AMPP partner logic functions, and           
# any output files any of the foregoing (including device programming           
# or simulation files), and any associated documentation or information         
# are expressly subject to the terms and conditions of the Altera               
# Program License Subscription Agreement, Intel MegaCore Function               
# License Agreement, or other applicable license agreement, including,          
# without limitation, that your use is for the sole purpose of                  
# programming logic devices manufactured by Intel and sold by Intel or          
# its authorized distributors.                                                  
# Please refer to the applicable agreement for further details.                 



=head1 SYNOPSIS

   use acl::Env;

   my $sdk_root = acl::Env::sdk_root();
   print "Did you set environment variable ". acl::Env::sdk_root_name() ." ?\n";

=cut

package acl::Env;
require Exporter;
@acl::Pkg::ISA        = qw(Exporter);
@acl::Pkg::EXPORT     = ();
@acl::Pkg::EXPORT_OK  = qw();
use strict;
require acl::Board_env;
require acl::File;
require acl::Common;
require acl::CommonData;

# checks host OS, returns true for linux, false otherwise.
sub is_linux() {
    if ($^O eq 'linux') {
      return 1;
    }
    return 0;
}

# checks for Windows host OS. Returns true if Windows, false otherwise.
sub is_windows() { return $^O =~ m/Win/; }
sub pathsep() { return is_windows() ? '\\' : '/'; }

# On Windows, always use 64-bit binaries.
# On Linux, always use 64-bit binaries, but via the wrapper shell scripts in "bin".
sub qbindir() { return is_windows() ? 'bin64' : 'bin'; }

# On what platform are we running?
# One of windows64, linux64 (Linux on x86-64), ppc64 (Linux on Power), arm32 (SoC)
sub get_arch {
   my ($arg) = shift @_;
   my $arch = $ENV{'AOCL_ARCH_OVERRIDE'};
   chomp $arch;
   if ($arg eq "--arm") {
      return 'arm32';
   } elsif ( ( $arch && $arch =~ m/Win/) || is_windows() ) {
      return 'windows64';
   } else {
      # Autodetect architecture.
      # Can override with an env var, for testability.  Matches shell wrapper.
      $arch = $arch || acl::Common::mybacktick('uname -m');
      chomp $arch;
      $ENV{'AOCL_ARCH_OVERRIDE'} = $arch; # Cache the result.
      if ( $arch =~ m/^x86_64/ ) {
         return 'linux64';
      } elsif ( $arch =~ m/^armv7l/ ) {
         return 'arm32';
      }
   }
   return undef;
}

sub get_icd_dir() {return is_windows() ? 'HKEY_LOCAL_MACHINE\\Software\\Khronos\\OpenCL\\Vendors' : '/etc/OpenCL/vendors' ;}
sub get_fcd_dir() {
  my $default_fcd_path = is_windows() ? 'HKEY_LOCAL_MACHINE\\Software\\Intel\\OpenCL\\Boards' : '/opt/Intel/OpenCL/Boards' ;
  my $fcd_path;
  if(defined $ENV{ACL_BOARD_VENDOR_PATH}) {
    my $user_defined_fcd_path = $ENV{ACL_BOARD_VENDOR_PATH};
    # resolve the trailling slashes
    $user_defined_fcd_path =~ s/\/$//;
    $fcd_path = is_windows()? 'HKEY_CURRENT_USER\\Software\\Intel\\OpenCL\\Boards' : $user_defined_fcd_path;
  } else {
    $fcd_path = $default_fcd_path;
  }
  return $fcd_path
}

sub sdk_root() { return $ENV{'INTELFPGAOCLSDKROOT'} || $ENV{'ACL_ROOT'}; }
sub sdk_root_name() { return 'INTELFPGAOCLSDKROOT'; }
sub sdk_root_shellname() {
   return (is_windows() ? '%' : '$' ). 'INTELFPGAOCLSDKROOT' . (is_windows() ? '%' : '' );
}
sub sdk_dev_bin_dir_shellname() {
   return
      sdk_root_shellname()
      . pathsep()
      . (is_windows() ? 'windows64' : 'linux64' )
      . pathsep()
      . 'bin';
}
sub sdk_dev_bin_dir() {
   return
      sdk_root()
      . pathsep()
      . (is_windows() ? 'windows64' : 'linux64' )
      . pathsep()
      . 'bin';
}
sub sdk_bin_dir() {
   return
      sdk_root()
      . pathsep()
      . 'bin';
}
sub enquote_if_windows {
  my ($path_to_enquote) = @_;
  if (is_windows()) {
    return qq{"$path_to_enquote"};
  }
  return $path_to_enquote;
}
sub sdk_host_bin_dir() { return join(pathsep(), sdk_root(), 'host', get_arch(), 'bin'); }
sub is_sdk() {  return -e sdk_dev_bin_dir(); }
sub sdk_aocl_exe() { return enquote_if_windows(sdk_bin_dir().pathsep().'aocl'); }
sub sdk_aoc_exe() { return enquote_if_windows(sdk_dev_bin_dir().pathsep().'aoc'); }
sub sdk_pkg_editor_exe() { return enquote_if_windows(sdk_host_bin_dir().pathsep().'aocl-binedit'); }
sub sdk_libedit_exe() { return enquote_if_windows(sdk_host_bin_dir().pathsep().'aocl-libedit'); }
sub sdk_boardspec_exe() { return enquote_if_windows(sdk_host_bin_dir().pathsep().'aocl-boardspec'); }
sub sdk_hash_exe() { return enquote_if_windows(sdk_host_bin_dir().pathsep().'aocl-hash'); }
sub sdk_extract_aocx_exe() { return enquote_if_windows(sdk_host_bin_dir().pathsep().'aocl-extract-aocx'); }
sub sdk_version() { return '2025.0.0.485db989daf205d1f3115c250809abcd4faf0211'; }

sub create_opencl_ipx($) {
  my $work_dir = shift;
  my $opencl_ipx = "$work_dir/opencl.ipx";
  open(my $fh, '>', $opencl_ipx) or die("Cannot open file $opencl_ipx $!");
  print $fh '<?xml version="1.0" encoding="UTF-8"?>
<library>
 <path path="${INTELFPGAOCLSDKROOT}/ip/*" />
</library>
';
  close $fh;
}

sub _get_host_compiler_type {
   # Returns 'msvc' or 'gnu' depending on argument selection.
   # Also return any remaining arguments after consuming an override switch.
   my @args = @_;
   my @return_args = ();
   my $is_msvc = ($^O =~ m/MSWin/i);
   my %msvc_arg = map { ($_,1) } qw( --msvc --windows );
   my %gnu_arg = map { ($_,1) } qw( --gnu --gcc --linux --arm );
   foreach my $arg ( @args ) {
      if ( $msvc_arg{$arg} ) { $is_msvc = 1; }
      elsif ( $gnu_arg{$arg} ) { $is_msvc = 0; }
      else { push @return_args, $arg; }
   }
   return ( $is_msvc ? 'msvc' : 'gnu' ), @return_args;
}

sub host_ldlibs(@) {
   # Return a string with the libraries to use,
   # followed by remaining arguments after consuming an override switch.
   my ($host_compiler_type,@args) = _get_host_compiler_type( @_ );
   acl::Board_env::override_platform($host_compiler_type);
   my $board_libs = acl::Board_env::get_xml_platform_tag("linklibs");
   my $result = '';
   my @return_args = ();
   foreach my $arg (@args) {
      push @return_args, $arg;
   }
   if ( $host_compiler_type eq 'msvc' ) {
      $result = "$board_libs alteracl.lib pkg_editor.lib libelf.lib";
   } else {
      my $host_arch = get_arch(@_);
      $result .= "-Wl,--no-as-needed -lalteracl";

      if (length $board_libs) {
         $result .= " $board_libs";
      }

      # We distribute libelf ourselves, which means it's not in a standard
      # search path and will be in a directory specified by one of the -L
      # options. Unfortunately, -L options are only used to find libraries
      # specifically given with -l options, and are NOT used when looking up
      # a library specified as a dependency (through a DT_NEEDED entry) of a
      # library specified with -l. Therefore until we start statically linking
      # to libelf, this option will have to be here (see case 222459).
      # UPDATE: Linux64 doesn't need this, since it will be picked up in runtime
      # from LD_LIBRARY_PATH. However, arm (c5soc) still fails without it. Probably
      # for similar reason that it needs lstdc++.
      $result .= " -lelf";

      # When using a cross compiler, the DT_NEEDED entry in libalteracl.so that
      # indicates that it depends on libstdc++.so does not seem to be enough
      # for the linker to actually find libstdc++.so. It does however work if
      # -lstdc++ is specified manually. It appears the reason is that the
      # list of search paths for -l options is different than for DT_NEEDED
      # entries. Specifically it seems like libraries from DT_NEEDED entries
      # are only searched for in the cross-compiler "sysroot", which seems to
      # not be where libstdc++ is. For now, just assume if the target
      # architecture is not linux64, than we are probably using a cross-compiler
      # (probably ARM).
      if ($host_arch ne 'linux64') {
         $result .= " -lstdc++";
      }
   }
   return $result,@return_args;
}


sub host_ldflags(@) {
   # Return a string with the linker flags to use (but not the list of libraries),
   # followed by remaining arguments after consuming an override switch.
   my ($host_compiler_type,@args) = _get_host_compiler_type( @_ );
   acl::Board_env::override_platform($host_compiler_type);
   my $board_flags = acl::Board_env::get_xml_platform_tag("linkflags");
   my $result = '';
   if ( $host_compiler_type eq 'msvc' ) {
      $result = "/libpath:".sdk_root()."/host/windows64/lib";
   } else {
      my $host_arch = get_arch(@_);
      $result = "$result-L".sdk_root()."/host/$host_arch/lib";
   }
   if ( ($board_flags ne '') && ($board_flags ne $result) ) { $result = $board_flags." ".$result; }
   return $result,@args;
}


sub host_link_config(@) {
   # Return a string with the link configuration, followed by remaining arguments.
   my ($host_compiler_type,@args) = _get_host_compiler_type( @_ );
   my ($ldflags) = host_ldflags(@_);
   my ($ldlibs, @return_args) = host_ldlibs(@_);
   return "$ldflags $ldlibs", @return_args;
}


sub set_board_override(@) {
   # Set the override board path
   acl::Board_env::set_board_path(@_);
}

sub board_path(@) {
   # Return a string with the board path,
   # followed by remaining arguments after consuming an override switch.
   return acl::Board_env::get_board_path(),@_;
}

sub aocl_boardspec(@) {
  my ($boardspec, $cmd ) = @_;
  $boardspec = acl::File::abs_path("$boardspec" . "/board_spec.xml") if ( -d $boardspec );
  if ( ! -f $boardspec ) {
    return "Error: Can't find board_spec.xml ($boardspec)";
  }
  my ($exe) = (sdk_boardspec_exe());
  my $result = acl::Common::mybacktick("$exe \"$boardspec\" $cmd");
  if( $result =~ /error/ ) {
    die( "Error: Parsing $boardspec ran into the following error:\n$result\n" );
  }
  return $result;
}

# Return a hash of name -> path
sub board_hw_list(@) {
  my (@args) = @_;
  my %boards = ();

  # We want to find $acl_board_path/*/*.xml, however acl::File::simple_glob
  # cannot handle the two-levels of wildcards. Do one at a time.
  my $acl_board_path = acl::Board_env::get_board_path();
  $acl_board_path .= "/";
  $acl_board_path .= acl::Board_env::get_hardware_dir();
  $acl_board_path = acl::File::abs_path($acl_board_path);
  my @board_dirs = acl::File::simple_glob($acl_board_path . "/*");
  foreach my $dir (@board_dirs) {
    my @board_spec = acl::File::simple_glob($dir . "/board_spec.xml");
    if(scalar(@board_spec) != 0) {
      my ($board) = aocl_boardspec($board_spec[0], "name");
      if ( defined $boards{ $board } ) {
        print "Error: Multiple boards named $board found at \n";
        print "Error:   $dir\n";
        print "Error:   $boards{ $board }\n";
        return (undef,@args);
      }
      $boards{ $board } = $dir; # E.g.: pac_s10 -> .../board/intel_s10sx_pac/hardware/pac_s10
    }
  }
  return (%boards,@args);
}

sub board_hw_path(@) {
   # Return a string with the path to a specified board variant,
   # followed by remaining arguments after consuming an override switch.
   my $variant = shift;
   my (%boards,@args) = board_hw_list(@_);
   if ( defined $boards{ $variant } ) {
     return ($boards{ $variant },@args);
   } else {
     # Maintain old behaviour - even if board doesn't exist, here is where
     # it would be
     my ($board_path,@args) = board_path(@args);
     my ($hwdir) = acl::Board_env::get_hardware_dir();
     return "$board_path/$hwdir/$variant",@args;
   }
}


sub board_mmdlib(@) {
   my ($host_compiler_type,@args) = _get_host_compiler_type( @_ );
   acl::Board_env::override_platform($host_compiler_type);
   my $board_libs = acl::Board_env::get_xml_platform_tag("mmdlib");
   return $board_libs,@args;
}

sub board_libs(@) {
   # Return a string with the libraries to compile a host program for the current board,
   # followed by remaining arguments after consuming an override switch.
   my ($host_compiler_type,@args) = _get_host_compiler_type( @_ );
   acl::Board_env::override_platform($host_compiler_type);
   my $board_libs = acl::Board_env::get_xml_platform_tag("linklibs");
   return $board_libs,@args;
}


sub board_link_flags(@) {
   # Return a string with the link flags to compile a host program for the current board,
   # followed by remaining arguments after consuming an override switch.
   my ($host_compiler_type,@args) = _get_host_compiler_type( @_ );
   acl::Board_env::override_platform($host_compiler_type);
   my $link_flags = acl::Board_env::get_xml_platform_tag("linkflags");
   return $link_flags,@args;
}


sub board_version(@) {
   # Return a string with the board version,
   # followed by remaining arguments after consuming an override switch.
   my @args = @_;
   my $board = acl::Board_env::get_board_version();
   return $board,@args;
}

sub board_name(@) {
   # Return a string with the board version,
   # followed by remaining arguments after consuming an override switch.
   my @args = @_;
   my $board = acl::Board_env::get_board_name();
   return $board,@args;
}

sub board_hardware_default(@) {
   # Return a string with the default board,
   # followed by remaining arguments after consuming an override switch.
   my @args = @_;
   my $board = acl::Board_env::get_hardware_default();
   return $board,@args;
}

sub board_mmdlib_if_exists() {
  # Return a string contains the mmdlibs. If no mmd found, return undef
  return acl::Board_env::get_mmdlib_if_exists();
}

sub board_post_qsys_script(@) {
  # Return a string with script to run after qsys. Sometimes needed to
  # fixup qsys output.
  return acl::Board_env::get_post_qsys_script();
}


# Extract Quartus version string from output of "quartus_sh --version"
# Returns {major => #, minor => #, update => #}.
#  E.g. "19.3.0" -> (major => 19, minor => 3, update => 0)
#  E.g. "19.3eap.0" -> (major => 19, minor => 3, update => 0)
#
# Returns undef if there is no version.
sub get_quartus_version($) {
   my $str = shift;
   if ( $str =~ /^(\d+)\.(\d+)([^\.]*)\.(\d+)$/ ) {
      # $3 is called variant, which is a special string for specially marked releases
      #  e.g. "19.3eap.0" -> variant = 'eap'
      # Variant has no effect on version check.
      #  e.g. acl/19.3eap.0 should work with acds/19.3.0 and vice-versa
      return {
        major => $1,
        minor => $2,
        update => $4,
      };
   }
   return undef;
}

# for SDL reasons, we want to be calling the full path of quartus_sh instead of trusting
# that it is on the PATH correctly (though the wrapper should have put it on the PATH)
sub get_quartus_sh_full_path {
  # the entry wrapper will set QUARTUS_ROOTDIR as long as it managed to find the quartus
  # lets trust whatever quartus it found is the right one!
  my $q_rootdir_setting = $ENV{"QUARTUS_ROOTDIR"};
  if ( $q_rootdir_setting && -d $q_rootdir_setting ) {
    if ( -d $q_rootdir_setting.q{/bin} || -d $q_rootdir_setting.q{/bin64}) {
      my $bindir = q{bin};
      if (! -d $q_rootdir_setting.q{/bin} ) {
        $bindir = q{bin64};
      }
      if ( is_windows() && -e $q_rootdir_setting.q{\\}.$bindir.q{\\quartus_sh.exe} ) {
        return $q_rootdir_setting.q{\\}.$bindir.q{\\quartus_sh.exe};
      } elsif ( !is_windows() && -e $q_rootdir_setting.qq{/$bindir/quartus_sh} ) {
        return $q_rootdir_setting.qq{/$bindir/quartus_sh};
      }
    }
  }

  # Didn't return by now - means we couldnt find the quartus full path
  # Exit, we didn't find quartus
  return "";
}

# Run "quartus_sh --version" to get the quartus version (optionally pass path to quartus_sh)
sub get_quartus_sh_version {
  my ($full_quartus_exe_path) = @_;

  my $quartus_sh_path = get_quartus_sh_full_path();
  my $cmd = "$quartus_sh_path --version";
  if ( ! -e $quartus_sh_path ) {
    # couldn't find quartus_sh here!
    return;
  }
  $ENV{QUARTUS_OPENCL_SDK}=1; #Tell Quartus that we are OpenCL
  my $q_out = acl::Common::mybacktick($cmd);
  if ($? != 0) { return; }

  chomp $q_out;
  return $q_out;
}

# Return 1 if the versions of the BSP using and quartus match
sub compare_quartus_bsp_version {
  # if $pro_std_only is set to 1, only do pro-std check, no need to do version # check
  my ($q_sh_version, $bsp_version, $need_quartus_pro, $need_quartus_std, $pro_std_only) = @_;

  my $quartus_version_local = q{};
  my $is_quartus_pro = 0;
  my $is_internal = 0;
  my $site = 0;
  my $q_ver = 0;
  foreach my $line (split ('\n', $q_sh_version)) {
    # Do version check.
    if ($line =~ m/Version (\S+)/) {
      my $qversion_str = $1;
      $quartus_version_local = acl::Env::get_quartus_version($qversion_str);
    }
    if ($line =~ /Pro Edition/) {
      $is_quartus_pro = 1;
    }
    if ($line =~ /Internal/) {
      $is_internal = 1;
    }
    # check which site it is from
    if ($line =~ m/\s+([A-Z][A-Z])\s+/) {
      $site = $1;
    }
  }
  if ( $quartus_version_local ) {
    $q_ver = $quartus_version_local->{major}.".".$quartus_version_local->{minor};
  }

  if ($is_quartus_pro && $need_quartus_std) {
    return { return_code => 0, quartus_version => $q_ver, is_internal => $is_internal, site => $site, is_std => !$is_quartus_pro };
  }
  if (!$is_quartus_pro && $need_quartus_pro) {
    return { return_code => 0, quartus_version => $q_ver, is_internal => $is_internal, site => $site, is_std => !$is_quartus_pro };
  }

  # early exit if we're only looking to do pro-std checks
  if ($pro_std_only) {
    return { return_code => 1, quartus_version => $q_ver, is_internal => $is_internal, site => $site, is_std => !$is_quartus_pro };
  }

  if ($bsp_version =~ m/(\d+).(\d+)/) {
    my $bsp_major = $1;
    my $bsp_minor = $2;
    if ( !$quartus_version_local ) {
      return { return_code => 0, quartus_version => q{}, is_internal => $is_internal, site => $site, is_std => !$is_quartus_pro };
    }

    if ( $bsp_major != $quartus_version_local->{major} || $bsp_minor != $quartus_version_local->{minor} ) {
      return { return_code => 0, quartus_version => $q_ver, is_internal => $is_internal, site => $site, is_std => !$is_quartus_pro };
    }

    return { return_code => 1, quartus_version => $q_ver, is_internal => $is_internal, site => $site, is_std => !$is_quartus_pro };
  } else {
    print "Cannot figure out the version of the BSP to confirm if Quartus version matches - assuming a match!\n" if acl::Common::get_verbose();
    return { return_code => 1, quartus_version => $q_ver, is_internal => $is_internal, site => $site, is_std => !$is_quartus_pro };
  }
}

# Look for quartus installation in the default install dir
# If find a quartus version in the supported range matching the BSP version, return it
# otherwise return a download link for a version that would be supported + match the BSP
sub find_a_better_quartus {
  my ($is_ipa, $bsp_version, $need_quartus_pro, $need_quartus_std) = @_;

  my $correct_q_ver = q{};
  my $sep = acl::Env::pathsep();
  my $install_dir_path = ($need_quartus_std) ? 'intelFPGA' : 'intelFPGA_pro';
  my $QUARTUS_PRO_DEFAULT_PATH_WIN = qq{C:\\$install_dir_path};
  my $home_variable = $ENV{q{HOME}};
  my $QUARTUS_PRO_DEFAULT_PATH_LIN = qq{$home_variable/$install_dir_path};
  my $quartus_default_install_dir = (is_windows()) ? $QUARTUS_PRO_DEFAULT_PATH_WIN : $QUARTUS_PRO_DEFAULT_PATH_LIN;
  if ( -d $quartus_default_install_dir ) {
    # iterate over directories in the default install path to go through all version #s here
    opendir( my $DEFAULT_INSTALL_DIR, $quartus_default_install_dir );
    while ( my $entry = readdir $DEFAULT_INSTALL_DIR ) {
      next unless -d  $quartus_default_install_dir.$sep.$entry;
      next if $entry eq "." or $entry eq "..";
      # found a subdirectory in the default install path - check if can find the quartus directory in it
      # if can, it's probably a installed quartus version
      if ( -d $quartus_default_install_dir.$sep.$entry.$sep.'quartus' ) {
        my $quartus_path = $quartus_default_install_dir.$sep.$entry.$sep.'quartus';
        my $quartus_bin = $quartus_path.$sep.qbindir();
        # check if the quartus bin directory does exist
        if ( -d $quartus_bin ) {
          my $q_out = acl::Env::get_quartus_sh_version($quartus_bin);
          if ( $q_out ) {
            # Found the quartus version - check if the version actually matches 
            my $compare_results = acl::Env::compare_quartus_bsp_version($q_out, $bsp_version, $need_quartus_pro, $need_quartus_std, $is_ipa);
            
            # check that this version is actually supported 
            if ( grep(/^$compare_results->{quartus_version}$/, @acl::CommonData::supported_acds_list) 
                && $compare_results->{return_code} ) {
              $correct_q_ver = acl::File::abs_path($quartus_bin);
              return ($correct_q_ver, 0);
            }
          }
        }
      }
    }
  }
  # Didn't find a better version already installed (at least far as we can tell)
  # in this case, return a download link to the the version they would need
  my $q_ver = (is_windows()) ? $bsp_version.'win' : $bsp_version.'lin';
  if ( $need_quartus_std ) {
    $q_ver = (is_windows()) ? 'stdwin' : 'stdlin';
  }
  my $quartus_download_link = $acl::CommonData::download_center_link;
  # For IPA, give them the link to the latest version instead since here quartus doesn't need to match
  if ($is_ipa) {
    my $q_ver;
    if ( $need_quartus_std ) {
      $q_ver = (is_windows()) ? 'stdwin' : 'stdlin';
    } else {
      $q_ver = (is_windows()) ? $acl::CommonData::latest_released_quartus.'win' : $acl::CommonData::latest_released_quartus.'lin';
    }
    if ( exists $acl::CommonData::quartus_download_links{$q_ver} ) {
      $quartus_download_link = $acl::CommonData::quartus_download_links{$q_ver};
    }
  } else {
    my $q_ver = (is_windows()) ? $bsp_version.'win' : $bsp_version.'lin';
    if ( $need_quartus_std ) {
      $q_ver = (is_windows()) ? 'stdwin' : 'stdlin';
    }
    if ( exists $acl::CommonData::quartus_download_links{$q_ver} ) {
      $quartus_download_link = $acl::CommonData::quartus_download_links{$q_ver};
    } else {
      # otherwise just return the download center link - means we couldn't find anything more specific
    }
  }
  return ($quartus_download_link, 1);
}


# Checks if the bsp version in the board_xml passed in is compatible with the aoc version.
# It is compatible if it is not newer and at oldest two releases older
# (eg. if the aoc version is 20.1, bsp versions 20.1, 19.3, and 19.1 are accepted)
# Versions 19.2 is the exception needed to for the DCP BSPs
sub is_bsp_version_compatible($) {
  my $bsp_version = shift;
  my $skd_version = sdk_version();
   if ( $skd_version =~ m{^
      (\d{2})(\d{2})             # Major (4 digits, split into two 2-digit pairs)
      [.](\d+)                   # Minor
      (\w+)?                     # Optional classification (alpha, beta,...)
      ([.]\d+)?                  # Optional patch version
      [.]([a-z\d]+)              # Build
      $}smx ) {
    my $aoc_version = "$2.$3";
    # if the bsp version is higher somehow or if it is more than three releases old, error out
    if (($aoc_version < $bsp_version) || ($aoc_version > $bsp_version + 3)) {
      # this exception is needed to use the a10pac and s10pac boards
      if ($bsp_version == 19.2) { return 1; }
      return 0;
    }
    return 1;
  }
  return 1; # something is wrong with the sdk version - can't do check, don't fail
}

1;
