# Copyright (C) 2023 Intel Corporation
# SPDX-License-Identifier: MIT

from dataclasses import dataclass, field
from pathlib import Path
from typing import NewType, Dict, Union

import pandas as pd

RawDataFrame = NewType('RawDataFrame', pd.DataFrame)

RawEmonDataFrame = NewType('RawEmonDataFrame', pd.DataFrame)


class RawDataFrameColumns:
    TIMESTAMP = 'timestamp'
    SOCKET = 'socket'
    DEVICE = 'device'
    CORE = 'core'
    THREAD = 'thread'
    UNIT = 'unit'
    MODULE = 'module'
    TSC = 'tsc'
    GROUP = 'group'
    NAME = 'name'
    VALUE = 'value'

    COLUMNS = [TIMESTAMP, SOCKET, DEVICE, CORE, THREAD, UNIT, MODULE, TSC, GROUP, NAME, VALUE]


class RawEmonDataFrameColumns(RawDataFrameColumns):
    pass


SummaryViewDataFrame = NewType('SummaryViewDataFrame', pd.DataFrame)


class SummaryViewDataFrameColumns:
    AGGREGATED = 'aggregated'
    MIN = 'min'
    MAX = 'max'
    PERCENTILE = '95th percentile'
    VARIATION = 'variation (stdev/avg)'
    DESCRIPTION = 'description'

    COLUMNS = [AGGREGATED, MIN, MAX, PERCENTILE, VARIATION, DESCRIPTION]


EventInfoDataFrame = NewType('EventInfoDataFrame', pd.DataFrame)


class EventInfoDataFrameColumns:
    NAME = 'name'
    DEVICE = 'device'


class RetireLatencyCountDataFrameColumn:
    COUNT = 'count'

    COLUMN = [COUNT]


RetireLatencyCountDataFrame = NewType('RetireLatencyCountDataFrame', pd.DataFrame)


@dataclass
class Threshold:
    metric_aliases: Dict[str, str]
    formula: str
    formula_raw: str


@dataclass(frozen=True)
class MetricDefinition:
    """
    Metric definition including all its attributes (name, description, formula, etc...).

    """
    # "current" name of the metric, i.e. the metric name
    name: str

    # corresponding metric name for "per transaction" metrics
    throughput_metric_name: str

    # metric description for documentation purposes
    description: str

    # the metric formula
    formula: str

    # maps event aliases (e.g. "a") to their respective event names (e.g. "INST_RETIRED.ANY")
    event_aliases: Dict[str, str] = field(default_factory=dict)

    # maps constant aliases (e.g. "a") to their value (e.g. "2", "system.sockets[0][0].size",
    # "$samplingTime", $processed_samples)
    constants: Dict[str, str] = field(default_factory=dict)

    # maps retire latency aliases (e.g. "a") to their respective latency names
    retire_latencies: Dict[str, str] = field(default_factory=dict)

    # the "standard" name of the metric
    canonical_name: str = ''

    # Level
    level: int = 0

    unit_of_measure: str = ''

    category: str = ''

    threshold: Threshold = ''

    resolution_levels: list = None

    metric_group: str = ''

    # a human-readable version of formula. For documentation purposes
    human_readable_expression: str = ''


class ConfigurationPaths:
    PATH: str = 'path'

    METRIC_PATH: str = f'metric {PATH}'
    CHART_PATH: str = f'chart {PATH}'
