/*
 * Copyright (C) 2011 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

#include "lwpmudrv_defines.h"

#include "lwpmudrv_types.h"
#include "rise_errors.h"
#include "lwpmudrv_ecb.h"
#include "lwpmudrv_struct.h"

#include "lwpmudrv.h"
#include "utility.h"
#include "control.h"
#include "output.h"
#include "silvermont.h"
#include "ecb_iterators.h"
#include "pebs.h"
#include "apic.h"

extern U64                       *read_counter_info;
extern DRV_CONFIG                 drv_cfg;
extern EMON_BUFFER_DRIVER_HELPER  emon_buffer_driver_helper;

#define ADD_ERRATA_FIX_FOR_FIXED_CTR0
#define MSR_ENERGY_MULTIPLIER         0x606        // Energy Multiplier MSR

#if defined(DRV_IA32)
#define ENABLE_IA32_PERFEVTSEL0_CTR 0x00400000
#define ENABLE_FIXED_CTR0           0x00000003
#elif defined(DRV_EM64T)
#define ENABLE_IA32_PERFEVTSEL0_CTR 0x0000000000400000
#define ENABLE_FIXED_CTR0           0x0000000000000003
#else
#error "Unexpected Architecture seen"
#endif

/* ------------------------------------------------------------------------- */
/*!
 * @fn void silvermont_Write_PMU(param)
 *
 * @param    param    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Initial set up of the PMU registers
 *
 * <I>Special Notes</I>
 *         Initial write of PMU registers.
 *         Walk through the enties and write the value of the register accordingly.
 *         Assumption:  For CCCR registers the enable bit is set to value 0.
 *         When current_group = 0, then this is the first time this routine is called,
 *         initialize the locks and set up EM tables.
 */
static VOID
silvermont_Write_PMU (
    VOID  *param
)
{
    U32            this_cpu = CONTROL_THIS_CPU();
    CPU_STATE      pcpu     = &pcb[this_cpu];
    U32            dev_idx  = core_to_dev_map[this_cpu];
    DEV_CONFIG     pcfg     = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    EVENT_CONFIG   ec       = LWPMU_DEVICE_ec(&devices[dev_idx]);
    DISPATCH       dispatch = LWPMU_DEVICE_dispatch(&devices[dev_idx]);

    if (!DEV_CONFIG_num_events(pcfg)) {
        return;
    }

    if (CPU_STATE_current_group(pcpu) == 0) {
        if (EVENT_CONFIG_mode(ec) != EM_DISABLED) {
            U32            index;
            U32            st_index;
            U32            j;

            /* Save all the initialization values away into an array for Event Multiplexing. */
            for (j = 0; j < EVENT_CONFIG_num_groups(ec); j++) {
                CPU_STATE_current_group(pcpu) = j;
                st_index   = CPU_STATE_current_group(pcpu) * EVENT_CONFIG_max_gp_events(ec);
                FOR_EACH_DATA_GP_REG(pecb, i) {
                    index = st_index + (ECB_entries_reg_id(pecb,i) - IA32_PMC0);
                    CPU_STATE_em_tables(pcpu)[index] = ECB_entries_reg_value(pecb,i);
                } END_FOR_EACH_DATA_GP_REG;
            }
            /* Reset the current group to the very first one. */
            CPU_STATE_current_group(pcpu) = this_cpu % EVENT_CONFIG_num_groups(ec);
        }
    }

    if (dispatch->hw_errata) {
        dispatch->hw_errata();
    }

    FOR_EACH_REG_ENTRY(pecb, i) {
        /*
         * Writing the GLOBAL Control register enables the PMU to start counting.
         * So write 0 into the register to prevent any counting from starting.
         */
        if (ECB_entries_reg_id(pecb,i) == IA32_PERF_GLOBAL_CTRL) {
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), 0LL);
            continue;
        }
        /*
         *  PEBS is enabled for this collection session
         */
        if (ECB_entries_reg_id(pecb,i) == IA32_PEBS_ENABLE &&
            ECB_entries_reg_value(pecb,i)) {
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), 0LL);
            continue;
        }
        SYS_Write_MSR(ECB_entries_reg_id(pecb,i), ECB_entries_reg_value(pecb,i));

    } END_FOR_EACH_REG_ENTRY;

#if defined ADD_ERRATA_FIX_FOR_FIXED_CTR0
    {
        U64 fixed_ctr0 = SYS_Read_MSR(IA32_FIXED_CTRL);
        fixed_ctr0 = (fixed_ctr0 & (ENABLE_FIXED_CTR0));
        if (fixed_ctr0 != 0x0) {
            U64 val = SYS_Read_MSR(IA32_PERFEVTSEL0);
            val |= ENABLE_IA32_PERFEVTSEL0_CTR;
            SYS_Write_MSR(IA32_PERFEVTSEL0, val);
        }
    }
#endif
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void silvermont_Disable_PMU(param)
 *
 * @param    param    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Zero out the global control register.  This automatically disables the PMU counters.
 *
 */
static VOID
silvermont_Disable_PMU (
    PVOID  param
)
{
    if (GET_DRIVER_STATE() != DRV_STATE_RUNNING) {
        SEP_DRV_LOG_TRACE("driver state = %d.", GET_DRIVER_STATE());
        SYS_Write_MSR(IA32_PERF_GLOBAL_CTRL, 0LL);
        SYS_Write_MSR(IA32_PEBS_ENABLE, 0LL);
        APIC_Disable_PMI();
    }

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void silvermont_Enable_PMU(param)
 *
 * @param    param    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Set the enable bit for all the Control registers
 *
 */
static VOID
silvermont_Enable_PMU (
    PVOID   param
)
{
    /*
     * Get the value from the event block
     *   0 == location of the global control reg for this block.
     *   Generalize this location awareness when possible
     */
    U32         this_cpu = CONTROL_THIS_CPU();
    CPU_STATE   pcpu     = &pcb[this_cpu];
    U32         dev_idx  = core_to_dev_map[this_cpu];
    U32         cur_grp  = CPU_STATE_current_group(pcpu);
    ECB         pecb     = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[cur_grp];

    if (!pecb) {
        return;
    }

    if (GET_DRIVER_STATE() == DRV_STATE_RUNNING) {
        APIC_Enable_Pmi();
        if (CPU_STATE_reset_mask(pcpu)) {
            SEP_DRV_LOG_TRACE("Overflow reset mask %llx.", CPU_STATE_reset_mask(pcpu));
            // Reinitialize the global overflow control register
            SYS_Write_MSR(IA32_PERF_GLOBAL_CTRL, ECB_entries_reg_value(pecb,0));
            SYS_Write_MSR(IA32_DEBUG_CTRL, ECB_entries_reg_value(pecb,3));
            CPU_STATE_reset_mask(pcpu) = 0LL;
        }
        if (CPU_STATE_group_swap(pcpu)) {
            CPU_STATE_group_swap(pcpu) = 0;
            SYS_Write_MSR(IA32_PERF_GLOBAL_CTRL, ECB_entries_reg_value(pecb,0));
            SYS_Write_MSR(IA32_PEBS_ENABLE, ECB_entries_reg_value(pecb,2));
            SYS_Write_MSR(IA32_DEBUG_CTRL, ECB_entries_reg_value(pecb,3));
        }
    }
    SEP_DRV_LOG_TRACE("Reenabled PMU with value 0x%llx.", ECB_entries_reg_value(pecb,0));

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn silvermont_Read_PMU_Data(param)
 *
 * @param    param    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Read all the data MSR's into a buffer.  Called by the interrupt handler.
 *
 */
static void
silvermont_Read_PMU_Data (
    PVOID   param,
    U32     dev_idx
)
{
    U32       j;
    U64      *buffer    = (U64 *)param;
    U32       this_cpu;
    CPU_STATE pcpu;
    U32       cur_grp;
    ECB       pecb;

    preempt_disable();
    this_cpu  = CONTROL_THIS_CPU();
    preempt_enable();
    pcpu      = &pcb[this_cpu];
    cur_grp   = CPU_STATE_current_group(pcpu);
    pecb      = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[cur_grp];
    j         = 0;
    if (!pecb) {
        return;
    }

    FOR_EACH_DATA_REG(pecb,i) {

        j = EMON_BUFFER_CORE_EVENT_OFFSET(EMON_BUFFER_DRIVER_HELPER_core_index_to_thread_offset_map(emon_buffer_driver_helper)[this_cpu],
                                          ECB_entries_core_event_id(pecb,i));

        buffer[j] = SYS_Read_MSR(ECB_entries_reg_id(pecb,i));
        SEP_DRV_LOG_TRACE("j=%u, value=%llu, cpu=%u, event_id=%u", j, buffer[j], this_cpu, ECB_entries_core_event_id(pecb,i));

    } END_FOR_EACH_DATA_REG;

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void silvermont_Check_Overflow(masks)
 *
 * @param    masks    the mask structure to populate
 *
 * @return   None     No return needed
 *
 * @brief  Called by the data processing method to figure out which registers have overflowed.
 *
 */
static void
silvermont_Check_Overflow (
    DRV_MASKS    masks
)
{
    U32              index;
    U64              overflow_status     = 0;
    U32              this_cpu            = CONTROL_THIS_CPU();
    BUFFER_DESC      bd                  = &cpu_buf[this_cpu];
    CPU_STATE        pcpu                = &pcb[this_cpu];
    U32              dev_idx             = core_to_dev_map[this_cpu];
    U32              cur_grp             = CPU_STATE_current_group(pcpu);
    ECB              pecb                = LWPMU_DEVICE_PMU_register_data(&devices[dev_idx])[cur_grp];
    DEV_CONFIG       pcfg                = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    DISPATCH         dispatch            = LWPMU_DEVICE_dispatch(&devices[dev_idx]);
    U64              overflow_status_clr = 0;
    DRV_EVENT_MASK_NODE event_flag;

    if (!pecb) {
        return;
    }

    // initialize masks
    DRV_MASKS_masks_num(masks) = 0;

    overflow_status = SYS_Read_MSR(IA32_PERF_GLOBAL_STATUS);

    if (DEV_CONFIG_pebs_mode(pcfg)) {
        overflow_status = PEBS_Overflowed (this_cpu, overflow_status);
    }
    overflow_status_clr = overflow_status;

    if (dispatch->check_overflow_gp_errata) {
        overflow_status = dispatch->check_overflow_gp_errata(pecb,  &overflow_status_clr);
    }
    SEP_DRV_LOG_TRACE("Overflow:  cpu: %d, status 0x%llx .", this_cpu, overflow_status);
    index                        = 0;
    BUFFER_DESC_sample_count(bd) = 0;
    FOR_EACH_DATA_REG(pecb, i) {
        if (ECB_entries_fixed_reg_get(pecb, i)) {
            index = ECB_entries_reg_id(pecb, i) - IA32_FIXED_CTR0 + 0x20;
            if (dispatch->check_overflow_errata) {
                overflow_status = dispatch->check_overflow_errata(pecb, i, overflow_status);
            }
        }
        else if (ECB_entries_is_gp_reg_get(pecb, i)) {
            index = ECB_entries_reg_id(pecb, i) - IA32_PMC0;
        }
        else {
            continue;
        }
        if (overflow_status & ((U64)1 << index)) {
            SEP_DRV_LOG_TRACE("Overflow:  cpu: %d, index %d.", this_cpu, index);
            SEP_DRV_LOG_TRACE("register 0x%x --- val 0%llx.",
                            ECB_entries_reg_id(pecb,i),
                            SYS_Read_MSR(ECB_entries_reg_id(pecb,i)));
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), ECB_entries_reg_value(pecb,i));

            DRV_EVENT_MASK_bitFields1(&event_flag) = (U8) 0;
            if (ECB_entries_fixed_reg_get(pecb, i)) {
                CPU_STATE_p_state_counting(pcpu) = 1;
            }
            if (ECB_entries_precise_get(pecb, i)) {
                DRV_EVENT_MASK_precise(&event_flag) = 1;
            }
            if (ECB_entries_lbr_value_get(pecb, i)) {
                DRV_EVENT_MASK_lbr_capture(&event_flag) = 1;
            }
            if (ECB_entries_uncore_get(pecb, i)) {
                DRV_EVENT_MASK_uncore_capture(&event_flag) = 1;
            }
            if (ECB_entries_em_trigger_get(pecb,i)) {
                DRV_EVENT_MASK_trigger(&event_flag) = 1;
            }

            if (DRV_MASKS_masks_num(masks) < MAX_OVERFLOW_EVENTS) {
                DRV_EVENT_MASK_bitFields1(DRV_MASKS_eventmasks(masks) + DRV_MASKS_masks_num(masks)) = DRV_EVENT_MASK_bitFields1(&event_flag);
                DRV_EVENT_MASK_event_idx(DRV_MASKS_eventmasks(masks) + DRV_MASKS_masks_num(masks)) = ECB_entries_event_id_index(pecb, i);
                DRV_EVENT_MASK_desc_id(DRV_MASKS_eventmasks(masks) + DRV_MASKS_masks_num(masks)) = ECB_entries_desc_id(pecb, i);
                DRV_MASKS_masks_num(masks)++;
            }
            else {
                SEP_DRV_LOG_ERROR("The array for event masks is full.");
            }

            SEP_DRV_LOG_TRACE("overflow -- 0x%llx, index 0x%llx.", overflow_status, (U64)1 << index);
            SEP_DRV_LOG_TRACE("slot# %d, reg_id 0x%x, index %d.",
                            i, ECB_entries_reg_id(pecb, i), index);
            if (ECB_entries_event_id_index(pecb, i) == CPU_STATE_trigger_event_num(pcpu)) {
                CPU_STATE_trigger_count(pcpu)--;
            }
        }
    } END_FOR_EACH_DATA_REG;


    CPU_STATE_reset_mask(pcpu) = overflow_status_clr;
    // Reinitialize the global overflow control register
    SYS_Write_MSR(IA32_PERF_GLOBAL_OVF_CTRL, overflow_status_clr);

    SEP_DRV_LOG_TRACE("Check Overflow completed %d.", this_cpu);
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn silvermont_Swap_Group(restart)
 *
 * @param    restart    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Perform the mechanics of swapping the event groups for event mux operations
 *
 * <I>Special Notes</I>
 *         Swap function for event multiplexing.
 *         Freeze the counting.
 *         Swap the groups.
 *         Enable the counting.
 *         Reset the event trigger count
 *
 */
static VOID
silvermont_Swap_Group (
    DRV_BOOL  restart
)
{
    U32            index;
    U32            next_group;
    U32            st_index;
    U32            this_cpu = CONTROL_THIS_CPU();
    CPU_STATE      pcpu     = &pcb[this_cpu];
    U32            dev_idx  = core_to_dev_map[this_cpu];
    DEV_CONFIG     pcfg     = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    EVENT_CONFIG   ec       = LWPMU_DEVICE_ec(&devices[dev_idx]);
    DISPATCH       dispatch = LWPMU_DEVICE_dispatch(&devices[dev_idx]);

    if (!DEV_CONFIG_num_events(pcfg)) {
        return;
    }

    st_index   = CPU_STATE_current_group(pcpu) * EVENT_CONFIG_max_gp_events(ec);
    next_group = (CPU_STATE_current_group(pcpu) + 1);
    if (next_group >= EVENT_CONFIG_num_groups(ec)) {
        next_group = 0;
    }

    SEP_DRV_LOG_TRACE("current group : 0x%x.", CPU_STATE_current_group(pcpu));
    SEP_DRV_LOG_TRACE("next group : 0x%x.", next_group);

    // Save the counters for the current group
    if (!DRV_CONFIG_event_based_counts(drv_cfg)) {
        FOR_EACH_DATA_GP_REG(pecb, i) {
            index = st_index + (ECB_entries_reg_id(pecb, i) - IA32_PMC0);
            CPU_STATE_em_tables(pcpu)[index] = SYS_Read_MSR(ECB_entries_reg_id(pecb,i));
            SEP_DRV_LOG_TRACE("Saved value for reg 0x%x : 0x%llx ",
                            ECB_entries_reg_id(pecb,i),
                            CPU_STATE_em_tables(pcpu)[index]);
        } END_FOR_EACH_DATA_GP_REG;
    }

    CPU_STATE_current_group(pcpu) = next_group;

    if (dispatch->hw_errata) {
        dispatch->hw_errata();
    }

    // First write the GP control registers (eventsel)
    FOR_EACH_CCCR_GP_REG(pecb, i) {
        SYS_Write_MSR(ECB_entries_reg_id(pecb,i), ECB_entries_reg_value(pecb,i));
    } END_FOR_EACH_CCCR_GP_REG;

    if (DRV_CONFIG_event_based_counts(drv_cfg)) {
        // In EBC mode, reset the counts for all events except for trigger event
        FOR_EACH_DATA_REG(pecb, i) {
            if (ECB_entries_event_id_index(pecb, i) != CPU_STATE_trigger_event_num(pcpu)) {
                SYS_Write_MSR(ECB_entries_reg_id(pecb,i), 0LL);
            }
        } END_FOR_EACH_DATA_REG;
    }
    else {
        // Then write the gp count registers
        st_index = CPU_STATE_current_group(pcpu) * EVENT_CONFIG_max_gp_events(ec);
        FOR_EACH_DATA_GP_REG(pecb, i) {
            index = st_index + (ECB_entries_reg_id(pecb, i) - IA32_PMC0);
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), CPU_STATE_em_tables(pcpu)[index]);
            SEP_DRV_LOG_TRACE("Restore value for reg 0x%x : 0x%llx ",
                            ECB_entries_reg_id(pecb,i),
                            CPU_STATE_em_tables(pcpu)[index]);
        } END_FOR_EACH_DATA_GP_REG;
    }

    FOR_EACH_ESCR_REG(pecb,i) {
        SYS_Write_MSR(ECB_entries_reg_id(pecb, i),ECB_entries_reg_value(pecb, i));
    } END_FOR_EACH_ESCR_REG;

    /*
     *  reset the em factor when a group is swapped
     */
    CPU_STATE_trigger_count(pcpu) = EVENT_CONFIG_em_factor(ec);

    /*
     * The enable routine needs to rewrite the control registers
     */
    CPU_STATE_reset_mask(pcpu) = 0LL;
    CPU_STATE_group_swap(pcpu) = 1;

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn silvermont_Initialize(params)
 *
 * @param    params    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Initialize the PMU setting up for collection
 *
 * <I>Special Notes</I>
 *         Saves the relevant PMU state (minimal set of MSRs required
 *         to avoid conflicts with other tools, such as hwpmc).
 *         This function should be called in parallel across all CPUs
 *         prior to the start of sampling, before PMU state is changed.
 *
 */
static VOID
silvermont_Initialize (
    VOID  *param
)
{
    U32        this_cpu = CONTROL_THIS_CPU();
    CPU_STATE  pcpu;

    SEP_DRV_LOG_TRACE("Inside silvermont_Initialize.");

    if (pcb == NULL) {
        return;
    }

    pcpu  = &pcb[this_cpu];
    CPU_STATE_pmu_state(pcpu) = pmu_state + (this_cpu * 2);
    if (CPU_STATE_pmu_state(pcpu) == NULL) {
        SEP_DRV_LOG_WARNING("Unable to save PMU state on CPU %d.",this_cpu);
        return;
    }

    // save the original PMU state on this CPU (NOTE: must only be called ONCE per collection)
    CPU_STATE_pmu_state(pcpu)[0] = SYS_Read_MSR(IA32_DEBUG_CTRL);
    CPU_STATE_pmu_state(pcpu)[1] = SYS_Read_MSR(IA32_PERF_GLOBAL_CTRL);

    SEP_DRV_LOG_TRACE("Saving PMU state on CPU %d :", this_cpu);
    SEP_DRV_LOG_TRACE("    msr_val(IA32_DEBUG_CTRL)=0x%llx .", CPU_STATE_pmu_state(pcpu)[0]);
    SEP_DRV_LOG_TRACE("    msr_val(IA32_PERF_GLOBAL_CTRL)=0x%llx .", CPU_STATE_pmu_state(pcpu)[1]);

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn silvermont_Destroy(params)
 *
 * @param    params    dummy parameter which is not used
 *
 * @return   None     No return needed
 *
 * @brief    Reset the PMU setting up after collection
 *
 * <I>Special Notes</I>
 *         Restores the previously saved PMU state done in core2_Initialize.
 *         This function should be called in parallel across all CPUs
 *         after sampling collection ends/terminates.
 *
 */
static VOID
silvermont_Destroy (
    VOID *param
)
{
    U32        this_cpu;
    CPU_STATE  pcpu;

    SEP_DRV_LOG_TRACE("Inside silvermont_Destroy.");

    if (pcb == NULL) {
        return;
    }

    preempt_disable();
    this_cpu = CONTROL_THIS_CPU();
    preempt_enable();
    pcpu = &pcb[this_cpu];

    if (CPU_STATE_pmu_state(pcpu) == NULL) {
        SEP_DRV_LOG_WARNING("Unable to restore PMU state on CPU %d.",this_cpu);
        return;
    }

    SEP_DRV_LOG_TRACE("Restoring PMU state on CPU %d :", this_cpu);
    SEP_DRV_LOG_TRACE("    msr_val(IA32_DEBUG_CTRL)=0x%llx .", CPU_STATE_pmu_state(pcpu)[0]);
    SEP_DRV_LOG_TRACE("    msr_val(IA32_PERF_GLOBAL_CTRL)=0x%llx .", CPU_STATE_pmu_state(pcpu)[1]);

    // restore the previously saved PMU state
    // (NOTE: assumes this is only called ONCE per collection)
    SYS_Write_MSR(IA32_DEBUG_CTRL, CPU_STATE_pmu_state(pcpu)[0]);
    SYS_Write_MSR(IA32_PERF_GLOBAL_CTRL, CPU_STATE_pmu_state(pcpu)[1]);

    CPU_STATE_pmu_state(pcpu) = NULL;

    return;
}

/*
 * @fn silvermont_Read_LBRs(buffer)
 *
 * @param   IN buffer - pointer to the buffer to write the data into
 * @return  None
 *
 * @brief   Read all the LBR registers into the buffer provided and return
 *
 */
static U64
silvermont_Read_LBRs (
    VOID   *buffer
)
{
    U32  i, count = 0;
    U64 *lbr_buf = (U64 *)buffer;
    U64 tos_ip_addr = 0;
    U64 tos_ptr = 0;
    U32 this_cpu = CONTROL_THIS_CPU();
    U32 dev_idx  = core_to_dev_map[this_cpu];
    LBR lbr      = LWPMU_DEVICE_lbr(&devices[dev_idx]);

    if (lbr == NULL) {
        return tos_ip_addr;
    }

    SEP_DRV_LOG_TRACE("Inside silvermont_Read_LBRs.");
    for (i = 0; i < LBR_num_entries(lbr); i++) {
        *lbr_buf = SYS_Read_MSR(LBR_entries_reg_id(lbr,i));
        SEP_DRV_LOG_TRACE("silvermont_Read_LBRs %u, 0x%llx.", i, *lbr_buf);
        if (i == 0) {
            tos_ptr = *lbr_buf;
        } else {
            if (LBR_entries_etype(lbr, i) == 1) {
                if (tos_ptr == count) {
                    tos_ip_addr = *lbr_buf;
                    SEP_DRV_LOG_TRACE("tos_ip_addr %llu, 0x%llx.", tos_ptr, *lbr_buf);
                }
                count++;
            }
        }
        lbr_buf++;
    }

    return tos_ip_addr;
}

static VOID
silvermont_Clean_Up (
    VOID   *param
)
{
    FOR_EACH_REG_ENTRY_IN_ALL_GRPS(pecb, i) {
        if (ECB_entries_clean_up_get(pecb,i)) {
            SEP_DRV_LOG_TRACE("clean up set --- RegId --- %x.", ECB_entries_reg_id(pecb,i));
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), 0LL);
        }
    } END_FOR_EACH_REG_ENTRY_IN_ALL_GRPS;

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn silvermont_Read_Counts(param, id)
 *
 * @param    param    The read thread node to process
 * @param    id       The event id for the which the sample is generated
 *
 * @return   None     No return needed
 *
 * @brief    Read CPU event based counts data and store into the buffer param;
 *           For the case of the trigger event, store the SAV value.
 */
static VOID
silvermont_Read_Counts (
    PVOID  param,
    U32    id
)
{
    U64            *data;
    U32             this_cpu     = CONTROL_THIS_CPU();
    CPU_STATE       pcpu         = &pcb[this_cpu];
    U32             dev_idx      = core_to_dev_map[this_cpu];
    DEV_CONFIG      pcfg         = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    U32             event_id     = 0;

    if (!DEV_CONFIG_num_events(pcfg)) {
        return;
    }

    if (DEV_CONFIG_ebc_group_id_offset(pcfg)) {
        // Write GroupID
        data  = (U64 *)((S8*)param + DEV_CONFIG_ebc_group_id_offset(pcfg));
        *data = CPU_STATE_current_group(pcpu) + 1;
    }

    FOR_EACH_DATA_REG(pecb,i) {
        if (ECB_entries_counter_event_offset(pecb,i) == 0) {
            continue;
        }
        data = (U64 *)((S8*)param + ECB_entries_counter_event_offset(pecb,i));
        event_id = ECB_entries_event_id_index(pecb,i);
        if (event_id == id) {
            *data = ~(ECB_entries_reg_value(pecb,i) - 1) &
                                           ECB_entries_max_bits(pecb,i);
        }
        else {
            *data = SYS_Read_MSR(ECB_entries_reg_id(pecb,i));
            SYS_Write_MSR(ECB_entries_reg_id(pecb,i), 0LL);
        }
    } END_FOR_EACH_DATA_REG;

    if (DRV_CONFIG_enable_p_state(drv_cfg)) {
        CPU_STATE_p_state_counting(pcpu) = 0;
    }

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID silvermont_Platform_Info
 *
 * @brief       Reads the MSR_PLATFORM_INFO register if present
 *
 * @param       void
 *
 * @return      value read from the register
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
static void
silvermont_Platform_Info (
    PVOID data
)
{
    DRV_PLATFORM_INFO      platform_data = (DRV_PLATFORM_INFO)data;
    U64                    index         = 0;
    U64                    value         = 0;
    U64                    clock_value   = 0;

    if (!platform_data) {
        return;
    }

#define IA32_MSR_PLATFORM_INFO 0xCE
    value = SYS_Read_MSR(IA32_MSR_PLATFORM_INFO);

#define IA32_MSR_PSB_CLOCK_STS  0xCD
#define FREQ_MASK_BITS          0x03

    clock_value = SYS_Read_MSR(IA32_MSR_PSB_CLOCK_STS);
    index = clock_value & FREQ_MASK_BITS;
    DRV_PLATFORM_INFO_info(platform_data)           = value;
    DRV_PLATFORM_INFO_ddr_freq_index(platform_data) = index;

#undef IA32_MSR_PLATFORM_INFO
#undef IA32_MSR_PSB_CLOCK_STS
#undef FREQ_MASK_BITS
    SEP_DRV_LOG_TRACE ("silvermont_Platform_Info: Read from MSR_ENERGY_MULTIPLIER reg is %d.", (U32)SYS_Read_MSR(MSR_ENERGY_MULTIPLIER));
    DRV_PLATFORM_INFO_energy_multiplier(platform_data) = (U32) (SYS_Read_MSR(MSR_ENERGY_MULTIPLIER) & 0x00001F00) >> 8;

    return;
}

/*
 * Initialize the dispatch table
 */
DISPATCH_NODE  silvermont_dispatch =
{
    silvermont_Initialize,       // init
    silvermont_Destroy,          // fini
    silvermont_Write_PMU,        // write
    silvermont_Disable_PMU,      // freeze
    silvermont_Enable_PMU,       // restart
    silvermont_Read_PMU_Data,    // read
    silvermont_Check_Overflow,   // check for overflow
    silvermont_Swap_Group,
    silvermont_Read_LBRs,
    silvermont_Clean_Up,
    NULL,
    NULL,
    NULL,
    silvermont_Read_Counts,
    NULL,
    NULL,                        // read_ro
    silvermont_Platform_Info,    // platform_info
    NULL,                        // scan_for_uncore
    NULL                         // read_metrics
};
