use crate::gobject_models::GTag;
use diffus::{Diffable, edit::Edit};
use eyre::{Result, eyre};
use glib::subclass::prelude::*;
use news_flash::models::{Tag, TagID};
use std::cmp::Ordering;
use std::collections::HashSet;

#[derive(Clone, Default, Debug)]
pub struct TagListModel {
    models: Vec<GTag>,
    tags: HashSet<TagID>,
}

impl TagListModel {
    pub fn new() -> Self {
        Self::default()
    }

    pub fn is_empty(&self) -> bool {
        self.models.is_empty()
    }

    pub fn add(&mut self, tag: Tag) -> Result<()> {
        if self.tags.contains(&tag.tag_id) {
            return Err(eyre!("tag model '{}' already exists in list", tag.tag_id));
        }
        let model: GTag = tag.into();
        self.tags.insert(model.tag_id().into());
        self.models.push(model);
        Ok(())
    }

    pub fn generate_diff<'a>(&'a self, other: &'a TagListModel) -> Edit<'a, Vec<GTag>> {
        self.models.diff(&other.models)
    }

    pub fn sort(&mut self) {
        self.models.sort_by(|a, b| {
            let imp_a = a.imp();
            let imp_b = b.imp();

            let sord_index_ordering = imp_a.sort_index.get().cmp(&imp_b.sort_index.get());
            if sord_index_ordering != Ordering::Equal {
                return sord_index_ordering;
            }

            let label_ordering = imp_a.label.borrow().as_str().cmp(imp_b.label.borrow().as_str());
            if label_ordering != Ordering::Equal {
                return label_ordering;
            }

            Ordering::Equal
        });
    }

    pub fn tags(&self) -> &[GTag] {
        &self.models
    }
}
