package builder

import (
	"os"
	"path/filepath"
	"strings"

	"github.com/tinygo-org/tinygo/goenv"
)

// libPicolibc is a C library for bare metal embedded devices. It was originally
// based on newlib.
var libPicolibc = Library{
	name: "picolibc",
	makeHeaders: func(target, includeDir string) error {
		f, err := os.Create(filepath.Join(includeDir, "picolibc.h"))
		if err != nil {
			return err
		}
		return f.Close()
	},
	cflags: func(target, headerPath string) []string {
		newlibDir := filepath.Join(goenv.Get("TINYGOROOT"), "lib/picolibc/newlib")
		return []string{
			"-Werror",
			"-Wall",
			"-std=gnu11",
			"-D_COMPILING_NEWLIB",
			"-D_HAVE_ALIAS_ATTRIBUTE",
			"-DTINY_STDIO",
			"-DPOSIX_IO",
			"-DFORMAT_DEFAULT_INTEGER", // use __i_vfprintf and __i_vfscanf by default
			"-D_IEEE_LIBM",
			"-D__OBSOLETE_MATH_FLOAT=1", // use old math code that doesn't expect a FPU
			"-D__OBSOLETE_MATH_DOUBLE=0",
			"-D_WANT_IO_C99_FORMATS",
			"-D__PICOLIBC_ERRNO_FUNCTION=__errno_location",
			"-nostdlibinc",
			"-isystem", newlibDir + "/libc/include",
			"-I" + newlibDir + "/libc/tinystdio",
			"-I" + newlibDir + "/libm/common",
			"-I" + headerPath,
		}
	},
	sourceDir: func() string { return filepath.Join(goenv.Get("TINYGOROOT"), "lib/picolibc/newlib") },
	librarySources: func(target string) ([]string, error) {
		sources := append([]string(nil), picolibcSources...)
		if !strings.HasPrefix(target, "avr") {
			// Small chips without long jumps can't compile many files (printf,
			// pow, etc). Therefore exclude those source files for those chips.
			// Unfortunately it's difficult to exclude only some chips, so this
			// excludes those files on all AVR chips for now.
			// More information:
			// https://github.com/llvm/llvm-project/issues/67042
			sources = append(sources, picolibcSourcesLarge...)
		}
		return sources, nil
	},
}

var picolibcSources = []string{
	"../../picolibc-stdio.c",

	"libc/string/bcmp.c",
	"libc/string/bcopy.c",
	"libc/string/bzero.c",
	"libc/string/explicit_bzero.c",
	"libc/string/ffsl.c",
	"libc/string/ffsll.c",
	"libc/string/fls.c",
	"libc/string/flsl.c",
	"libc/string/flsll.c",
	"libc/string/gnu_basename.c",
	"libc/string/index.c",
	"libc/string/memccpy.c",
	"libc/string/memchr.c",
	"libc/string/memcmp.c",
	"libc/string/memcpy.c",
	"libc/string/memmem.c",
	"libc/string/memmove.c",
	"libc/string/mempcpy.c",
	"libc/string/memrchr.c",
	"libc/string/memset.c",
	"libc/string/rawmemchr.c",
	"libc/string/rindex.c",
	"libc/string/stpcpy.c",
	"libc/string/stpncpy.c",
	"libc/string/strcasecmp.c",
	"libc/string/strcasecmp_l.c",
	"libc/string/strcasestr.c",
	"libc/string/strcat.c",
	"libc/string/strchr.c",
	"libc/string/strchrnul.c",
	"libc/string/strcmp.c",
	"libc/string/strcoll.c",
	"libc/string/strcoll_l.c",
	"libc/string/strcpy.c",
	"libc/string/strcspn.c",
	"libc/string/strdup.c",
	"libc/string/strerror.c",
	"libc/string/strerror_r.c",
	"libc/string/strlcat.c",
	"libc/string/strlcpy.c",
	"libc/string/strlen.c",
	"libc/string/strlwr.c",
	"libc/string/strncasecmp.c",
	"libc/string/strncasecmp_l.c",
	"libc/string/strncat.c",
	"libc/string/strncmp.c",
	"libc/string/strncpy.c",
	"libc/string/strndup.c",
	"libc/string/strnlen.c",
	"libc/string/strnstr.c",
	"libc/string/strpbrk.c",
	"libc/string/strrchr.c",
	"libc/string/strsep.c",
	"libc/string/strsignal.c",
	"libc/string/strspn.c",
	"libc/string/strstr.c",
	"libc/string/strtok.c",
	"libc/string/strtok_r.c",
	"libc/string/strupr.c",
	"libc/string/strverscmp.c",
	"libc/string/strxfrm.c",
	"libc/string/strxfrm_l.c",
	"libc/string/swab.c",
	"libc/string/timingsafe_bcmp.c",
	"libc/string/timingsafe_memcmp.c",
	"libc/string/u_strerr.c",
	"libc/string/wcpcpy.c",
	"libc/string/wcpncpy.c",
	"libc/string/wcscasecmp.c",
	"libc/string/wcscasecmp_l.c",
	"libc/string/wcscat.c",
	"libc/string/wcschr.c",
	"libc/string/wcscmp.c",
	"libc/string/wcscoll.c",
	"libc/string/wcscoll_l.c",
	"libc/string/wcscpy.c",
	"libc/string/wcscspn.c",
	"libc/string/wcsdup.c",
	"libc/string/wcslcat.c",
	"libc/string/wcslcpy.c",
	"libc/string/wcslen.c",
	"libc/string/wcsncasecmp.c",
	"libc/string/wcsncasecmp_l.c",
	"libc/string/wcsncat.c",
	"libc/string/wcsncmp.c",
	"libc/string/wcsncpy.c",
	"libc/string/wcsnlen.c",
	"libc/string/wcspbrk.c",
	"libc/string/wcsrchr.c",
	"libc/string/wcsspn.c",
	"libc/string/wcsstr.c",
	"libc/string/wcstok.c",
	"libc/string/wcswidth.c",
	"libc/string/wcsxfrm.c",
	"libc/string/wcsxfrm_l.c",
	"libc/string/wcwidth.c",
	"libc/string/wmemchr.c",
	"libc/string/wmemcmp.c",
	"libc/string/wmemcpy.c",
	"libc/string/wmemmove.c",
	"libc/string/wmempcpy.c",
	"libc/string/wmemset.c",
	"libc/string/xpg_strerror_r.c",
}

// Parts of picolibc that are too large for small AVRs.
var picolibcSourcesLarge = []string{
	// srcs_tinystdio
	"libc/tinystdio/asprintf.c",
	"libc/tinystdio/bufio.c",
	"libc/tinystdio/clearerr.c",
	"libc/tinystdio/ecvt_r.c",
	"libc/tinystdio/ecvt.c",
	"libc/tinystdio/ecvtf_r.c",
	"libc/tinystdio/ecvtf.c",
	"libc/tinystdio/fcvt.c",
	"libc/tinystdio/fcvt_r.c",
	"libc/tinystdio/fcvtf.c",
	"libc/tinystdio/fcvtf_r.c",
	"libc/tinystdio/gcvt.c",
	"libc/tinystdio/gcvtf.c",
	"libc/tinystdio/fclose.c",
	"libc/tinystdio/fdevopen.c",
	"libc/tinystdio/feof.c",
	"libc/tinystdio/ferror.c",
	"libc/tinystdio/fflush.c",
	"libc/tinystdio/fgetc.c",
	"libc/tinystdio/fgets.c",
	"libc/tinystdio/fileno.c",
	"libc/tinystdio/filestrget.c",
	"libc/tinystdio/filestrput.c",
	"libc/tinystdio/filestrputalloc.c",
	"libc/tinystdio/fmemopen.c",
	"libc/tinystdio/fprintf.c",
	"libc/tinystdio/fputc.c",
	"libc/tinystdio/fputs.c",
	"libc/tinystdio/fread.c",
	//"libc/tinystdio/freopen.c", // crashes with AVR, see: https://github.com/picolibc/picolibc/pull/369
	"libc/tinystdio/fscanf.c",
	"libc/tinystdio/fseek.c",
	"libc/tinystdio/fseeko.c",
	"libc/tinystdio/ftell.c",
	"libc/tinystdio/ftello.c",
	"libc/tinystdio/fwrite.c",
	"libc/tinystdio/getchar.c",
	"libc/tinystdio/gets.c",
	"libc/tinystdio/matchcaseprefix.c",
	"libc/tinystdio/mktemp.c",
	"libc/tinystdio/perror.c",
	"libc/tinystdio/printf.c",
	"libc/tinystdio/putchar.c",
	"libc/tinystdio/puts.c",
	"libc/tinystdio/rewind.c",
	"libc/tinystdio/scanf.c",
	"libc/tinystdio/setbuf.c",
	"libc/tinystdio/setbuffer.c",
	"libc/tinystdio/setlinebuf.c",
	"libc/tinystdio/setvbuf.c",
	"libc/tinystdio/snprintf.c",
	"libc/tinystdio/sprintf.c",
	"libc/tinystdio/snprintfd.c",
	"libc/tinystdio/snprintff.c",
	"libc/tinystdio/sprintff.c",
	"libc/tinystdio/sprintfd.c",
	"libc/tinystdio/sscanf.c",
	"libc/tinystdio/strfromf.c",
	"libc/tinystdio/strfromd.c",
	"libc/tinystdio/strtof.c",
	"libc/tinystdio/strtof_l.c",
	"libc/tinystdio/strtod.c",
	"libc/tinystdio/strtod_l.c",
	"libc/tinystdio/ungetc.c",
	"libc/tinystdio/vasprintf.c",
	"libc/tinystdio/vfiprintf.c",
	"libc/tinystdio/vfprintf.c",
	"libc/tinystdio/vfprintff.c",
	"libc/tinystdio/vfscanf.c",
	"libc/tinystdio/vfiscanf.c",
	"libc/tinystdio/vfscanff.c",
	"libc/tinystdio/vprintf.c",
	"libc/tinystdio/vscanf.c",
	"libc/tinystdio/vsscanf.c",
	"libc/tinystdio/vsnprintf.c",
	"libc/tinystdio/vsprintf.c",

	"libm/common/sf_finite.c",
	"libm/common/sf_copysign.c",
	"libm/common/sf_modf.c",
	"libm/common/sf_scalbn.c",
	"libm/common/sf_cbrt.c",
	"libm/common/sf_exp10.c",
	"libm/common/sf_expm1.c",
	"libm/common/sf_ilogb.c",
	"libm/common/sf_infinity.c",
	"libm/common/sf_isinf.c",
	"libm/common/sf_isinff.c",
	"libm/common/sf_isnan.c",
	"libm/common/sf_isnanf.c",
	"libm/common/sf_issignaling.c",
	"libm/common/sf_log1p.c",
	"libm/common/sf_nan.c",
	"libm/common/sf_nextafter.c",
	"libm/common/sf_pow10.c",
	"libm/common/sf_rint.c",
	"libm/common/sf_logb.c",
	"libm/common/sf_fdim.c",
	"libm/common/sf_fma.c",
	"libm/common/sf_fmax.c",
	"libm/common/sf_fmin.c",
	"libm/common/sf_fpclassify.c",
	"libm/common/sf_lrint.c",
	"libm/common/sf_llrint.c",
	"libm/common/sf_lround.c",
	"libm/common/sf_llround.c",
	"libm/common/sf_nearbyint.c",
	"libm/common/sf_remquo.c",
	"libm/common/sf_round.c",
	"libm/common/sf_scalbln.c",
	"libm/common/sf_trunc.c",
	"libm/common/sf_exp.c",
	"libm/common/sf_exp2.c",
	"libm/common/sf_exp2_data.c",
	"libm/common/sf_log.c",
	"libm/common/sf_log_data.c",
	"libm/common/sf_log2.c",
	"libm/common/sf_log2_data.c",
	"libm/common/sf_pow_log2_data.c",
	"libm/common/sf_pow.c",

	"libm/common/s_finite.c",
	"libm/common/s_copysign.c",
	"libm/common/s_modf.c",
	"libm/common/s_scalbn.c",
	"libm/common/s_cbrt.c",
	"libm/common/s_exp10.c",
	"libm/common/s_expm1.c",
	"libm/common/s_ilogb.c",
	"libm/common/s_infinity.c",
	"libm/common/s_iseqsig.c",
	"libm/common/s_isinf.c",
	"libm/common/s_isinfd.c",
	"libm/common/s_isnan.c",
	"libm/common/s_isnand.c",
	"libm/common/s_issignaling.c",
	"libm/common/s_log1p.c",
	"libm/common/s_nan.c",
	"libm/common/s_nextafter.c",
	"libm/common/s_pow10.c",
	"libm/common/s_rint.c",
	"libm/common/s_logb.c",
	"libm/common/s_log2.c",
	"libm/common/s_fdim.c",
	"libm/common/s_fma.c",
	"libm/common/s_fmax.c",
	"libm/common/s_fmin.c",
	"libm/common/s_fpclassify.c",
	"libm/common/s_getpayload.c",
	"libm/common/s_lrint.c",
	"libm/common/s_llrint.c",
	"libm/common/s_lround.c",
	"libm/common/s_llround.c",
	"libm/common/s_nearbyint.c",
	"libm/common/s_remquo.c",
	"libm/common/s_round.c",
	"libm/common/s_scalbln.c",
	"libm/common/s_signbit.c",
	"libm/common/s_trunc.c",
	"libm/common/exp.c",
	"libm/common/exp2.c",
	"libm/common/exp_data.c",
	"libm/common/math_err_with_errno.c",
	"libm/common/math_err_uflow.c",
	"libm/common/math_err_oflow.c",
	"libm/common/math_err_divzero.c",
	"libm/common/math_err_invalid.c",
	"libm/common/math_err_may_uflow.c",
	"libm/common/math_err_check_uflow.c",
	"libm/common/math_err_check_oflow.c",
	"libm/common/math_errf_divzerof.c",
	"libm/common/math_errf_invalidf.c",
	"libm/common/math_errf_may_uflowf.c",
	"libm/common/math_errf_oflowf.c",
	"libm/common/math_errf_uflowf.c",
	"libm/common/math_errf_with_errnof.c",
	"libm/common/math_inexact.c",
	"libm/common/math_inexactf.c",
	"libm/common/log.c",
	"libm/common/log_data.c",
	"libm/common/log2.c",
	"libm/common/log2_data.c",
	"libm/common/pow.c",
	"libm/common/pow_log_data.c",

	"libm/math/k_cos.c",
	"libm/math/k_rem_pio2.c",
	"libm/math/k_sin.c",
	"libm/math/k_tan.c",
	"libm/math/kf_cos.c",
	"libm/math/kf_rem_pio2.c",
	"libm/math/kf_sin.c",
	"libm/math/kf_tan.c",
	"libm/math/s_acos.c",
	"libm/math/s_acosh.c",
	"libm/math/s_asin.c",
	"libm/math/s_asinh.c",
	"libm/math/s_atan.c",
	"libm/math/s_atan2.c",
	"libm/math/s_atanh.c",
	"libm/math/s_ceil.c",
	"libm/math/s_cos.c",
	"libm/math/s_cosh.c",
	"libm/math/s_drem.c",
	"libm/math/s_erf.c",
	"libm/math/s_exp.c",
	"libm/math/s_exp2.c",
	"libm/math/s_fabs.c",
	"libm/math/s_floor.c",
	"libm/math/s_fmod.c",
	"libm/math/s_frexp.c",
	"libm/math/s_gamma.c",
	"libm/math/s_hypot.c",
	"libm/math/s_j0.c",
	"libm/math/s_j1.c",
	"libm/math/s_jn.c",
	"libm/math/s_lgamma.c",
	"libm/math/s_log.c",
	"libm/math/s_log10.c",
	"libm/math/s_pow.c",
	"libm/math/s_rem_pio2.c",
	"libm/math/s_remainder.c",
	"libm/math/s_scalb.c",
	"libm/math/s_signif.c",
	"libm/math/s_sin.c",
	"libm/math/s_sincos.c",
	"libm/math/s_sinh.c",
	"libm/math/s_sqrt.c",
	"libm/math/s_tan.c",
	"libm/math/s_tanh.c",
	"libm/math/s_tgamma.c",
	"libm/math/sf_acos.c",
	"libm/math/sf_acosh.c",
	"libm/math/sf_asin.c",
	"libm/math/sf_asinh.c",
	"libm/math/sf_atan.c",
	"libm/math/sf_atan2.c",
	"libm/math/sf_atanh.c",
	"libm/math/sf_ceil.c",
	"libm/math/sf_cos.c",
	"libm/math/sf_cosh.c",
	"libm/math/sf_drem.c",
	"libm/math/sf_erf.c",
	"libm/math/sf_exp.c",
	"libm/math/sf_exp2.c",
	"libm/math/sf_fabs.c",
	"libm/math/sf_floor.c",
	"libm/math/sf_fmod.c",
	"libm/math/sf_frexp.c",
	"libm/math/sf_gamma.c",
	"libm/math/sf_hypot.c",
	"libm/math/sf_j0.c",
	"libm/math/sf_j1.c",
	"libm/math/sf_jn.c",
	"libm/math/sf_lgamma.c",
	"libm/math/sf_log.c",
	"libm/math/sf_log10.c",
	"libm/math/sf_log2.c",
	"libm/math/sf_pow.c",
	"libm/math/sf_rem_pio2.c",
	"libm/math/sf_remainder.c",
	"libm/math/sf_scalb.c",
	"libm/math/sf_signif.c",
	"libm/math/sf_sin.c",
	"libm/math/sf_sincos.c",
	"libm/math/sf_sinh.c",
	"libm/math/sf_sqrt.c",
	"libm/math/sf_tan.c",
	"libm/math/sf_tanh.c",
	"libm/math/sf_tgamma.c",
	"libm/math/sr_lgamma.c",
	"libm/math/srf_lgamma.c",
}
