!===============================================================================
! Copyright (C) 2021 Intel Corporation
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

! Content:
! A simple example of single-precision complex-to-complex in-place 3D
! FFT using Intel(R) oneAPI Math Kernel Library (oneMKL) DFTI
!
!*****************************************************************************

include "mkl_dfti_omp_offload.f90"

program sp_complex_3d
  use MKL_DFTI_OMP_OFFLOAD, forget => DFTI_SINGLE, DFTI_SINGLE => DFTI_SINGLE_R
  use omp_lib, ONLY : omp_get_num_devices
  use, intrinsic :: ISO_C_BINDING

  ! Sizes of 3D transform
  integer, parameter :: N1 = 7
  integer, parameter :: N2 = 13
  integer, parameter :: N3 = 5

  ! Arbitrary harmonic used to test the FFT
  integer, parameter :: H1 = 1
  integer, parameter :: H2 = 2
  integer, parameter :: H3 = 3

  ! Need single precision
  integer, parameter :: WP = selected_real_kind(6,37)

  ! Execution status
  integer :: status = 0, ignored_status

  ! Data array
  complex(WP), allocatable :: x (:)

  ! DFTI descriptor handle
  type(DFTI_DESCRIPTOR), POINTER :: hand

  hand => null()

  print *,"Example sp_complex_3d"
  print *,"Forward and backward single-precision complex-to-complex",        &
    &      " in-place 3D transform"
  print *,"Configuration parameters:"
  print *,"DFTI_PRECISION      = DFTI_SINGLE"
  print *,"DFTI_FORWARD_DOMAIN = DFTI_COMPLEX"
  print *,"DFTI_DIMENSION      = 3"
  print '(" DFTI_LENGTHS        = /"I0","I0","I0"/" )', N1, N2, N3

  print *,"Create DFTI descriptor"
  status = DftiCreateDescriptor(hand, DFTI_SINGLE, DFTI_COMPLEX, 3, [N1,N2,N3])
  if (0 /= status) goto 999

  print *,"Commit DFTI descriptor"
  !$omp dispatch
  status = DftiCommitDescriptor(hand)
  if (0 /= status) goto 999

  print *,"Allocate array for in-place FFT"
  allocate ( x(N1*N2*N3), STAT = status)
  if (0 /= status) goto 999

  print *,"Initialize input for forward transform"
  call init(x, N1, N2, N3, H1, H2, H3)

  print *,"Compute forward transform"
  !$omp target data map(tofrom:x)
  !$omp dispatch
  status = DftiComputeForward(hand, x)
  !$omp end target data
  if (0 /= status) goto 999

  print *,"Verify the result"
  status = verify(x, N1, N2, N3, H1, H2, H3)
  if (0 /= status) goto 999

  print *,"Initialize input for backward transform"
  call init(x, N1, N2, N3, -H1, -H2, -H3)

  print *,"Compute backward transform"
  !$omp target data map(tofrom:x)
  !$omp dispatch
  status = DftiComputeBackward(hand, x)
  !$omp end target data
  if (0 /= status) goto 999

  print *,"Verify the result"
  status = verify(x, N1, N2, N3, H1, H2, H3)
  if (0 /= status) goto 999

100 continue

  print *,"Release the DFTI descriptor"
  ignored_status = DftiFreeDescriptor(hand)

  if (allocated(x)) then
      print *,"Deallocate data array"
      deallocate(x)
  endif

  if (status == 0) then
    print *,"TEST PASSED"
    call exit(0)
  else
    print *,"TEST FAILED"
    call exit(1)
  endif

999 print '("  Error, status = ",I0)', status
  goto 100

contains

  ! Compute mod(K*L,M) accurately
  pure real(WP) function moda(k,l,m)
    integer, intent(in) :: k,l,m
    integer*8 :: k8
    k8 = k
    moda = real(mod(k8*l,m),WP)
  end function moda

  ! Initialize array with harmonic /H1, H2, H3/
  subroutine init(x, N1, N2, N3, H1, H2, H3)
    integer N1, N2, N3, H1, H2, H3
    complex(WP) :: x(:)

    integer k1, k2, k3
    complex(WP), parameter :: I_TWOPI = (0.0_WP,6.2831853071795864769_WP)

    forall (k1=1:N1, k2=1:N2, k3=1:N3)
      x((k3-1)*N2*N1+(k2-1)*N1+k1) = exp( I_TWOPI*&
      ( moda(k1-1,H1,N1)/N1 &
      + moda(k2-1,H2,N2)/N2 &
      + moda(k3-1,H3,N3)/N3 ))/(N1*N2*N3)
    end forall
  end subroutine init

  ! Verify that x is unit peak at /H1, H2, H3/
  integer function verify(x, N1, N2, N3, H1, H2, H3)
    integer N1, N2, N3, H1, H2, H3
    complex(WP) :: x(:)

    integer k1, k2, k3
    real(WP) err, errthr, maxerr
    complex(WP) res_exp, res_got

    ! Note, this simple error bound doesn't take into account error of
    ! input data
    errthr = 5.0 * log(real(N1*N2*N3,WP)) / log(2.0_WP) * EPSILON(1.0_WP)
    print '("  Check if err is below errthr " G10.3)', errthr

    maxerr = 0.0_WP
    do k3 = 1, N3
      do k2 = 1, N2
        do k1 = 1, N1
          if (mod(k1-1-H1,N1)==0 .AND.                                  &
              mod(k2-1-H2,N2)==0 .AND.                                  &
              mod(k3-1-H3,N3)==0) then
            res_exp = 1.0_WP
          else
            res_exp = 0.0_WP
          end if
          res_got = x((k3-1)*N2*N1+(k2-1)*N1+k1)
          err = abs(res_got - res_exp)
          maxerr = max(err,maxerr)
          if (.not.(err < errthr)) then
            print '("  x("I0","I0","I0"):"$)', k1, k2, k3
            print '(" expected ("G14.7","G14.7"),"$)', res_exp
            print '(" got ("G14.7","G14.7"),"$)', res_got
            print '(" err "G10.3)', err
            print *," Verification FAILED"
            verify = 100
            return
          end if
        end do
      end do
    end do
    print '("  Verified,  maximum error was " G10.3)', maxerr
    verify = 0
  end function verify

end program sp_complex_3d
