/*
 * SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * Copyright (c) 2009 Mathieu Desnoyers <mathieu.desnoyers@efficios.com>
 *
 * Userspace RCU library - sys_futex compatibility code
 */

#include <stdio.h>
#include <pthread.h>
#include <signal.h>
#include <assert.h>
#include <errno.h>
#include <poll.h>
#include <stdint.h>

#include <urcu/arch.h>
#include <urcu/system.h>

#include "lib/lttng-ust/futex.h"

/*
 * This compat header originated in userspace RCU where it's used across
 * multiple shared objects hence the need to have the mutexes as public weak
 * symbols, in our case here, it's only used internally by liblttng-ust so we
 * can hide them. If we end up using this compat header in another library in
 * this project we will have to use the same scheme, but in the meantime, don't
 * expose those symbols in the ABI.
 */

/*
 * This comment will apply if we start using this compat header in multiple
 * libraires.
 *
 * Using attribute "weak" for __lttng_ust_compat_futex_lock and
 * __lttng_ust_compat_futex_cond. Those are globally visible by the entire
 * program, even though many shared objects may have their own version.
 * The first version that gets loaded will be used by the entire program
 * (executable and all shared objects).
 */

static pthread_mutex_t __lttng_ust_compat_futex_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t __lttng_ust_compat_futex_cond = PTHREAD_COND_INITIALIZER;

/*
 * _NOT SIGNAL-SAFE_. pthread_cond is not signal-safe anyway. Though.
 * For now, timeout, uaddr2 and val3 are unused.
 * Waiter will relinquish the CPU until woken up.
 */

int lttng_ust_compat_futex_noasync(int32_t *uaddr, int op, int32_t val,
	const struct timespec *timeout, int32_t *uaddr2, int32_t val3)
{
	int ret = 0, lockret;

	/*
	 * Check if NULL. Don't let users expect that they are taken into
	 * account.
	 */
	assert(!timeout);
	assert(!uaddr2);
	assert(!val3);

	/*
	 * memory barriers to serialize with the previous uaddr modification.
	 */
	cmm_smp_mb();

	lockret = pthread_mutex_lock(&__lttng_ust_compat_futex_lock);
	if (lockret) {
		errno = lockret;
		ret = -1;
		goto end;
	}
	switch (op) {
	case FUTEX_WAIT:
		/*
		 * Wait until *uaddr is changed to something else than "val".
		 * Comparing *uaddr content against val figures out which
		 * thread has been awakened.
		 */
		while (CMM_LOAD_SHARED(*uaddr) == val)
			pthread_cond_wait(&__lttng_ust_compat_futex_cond,
				&__lttng_ust_compat_futex_lock);
		break;
	case FUTEX_WAKE:
		/*
		 * Each wake is sending a broadcast, thus attempting wakeup of
		 * all awaiting threads, independently of their respective
		 * uaddr.
		 */
		pthread_cond_broadcast(&__lttng_ust_compat_futex_cond);
		break;
	default:
		errno = EINVAL;
		ret = -1;
	}
	lockret = pthread_mutex_unlock(&__lttng_ust_compat_futex_lock);
	if (lockret) {
		errno = lockret;
		ret = -1;
	}
end:
	return ret;
}

/*
 * _ASYNC SIGNAL-SAFE_.
 * For now, timeout, uaddr2 and val3 are unused.
 * Waiter will busy-loop trying to read the condition.
 * It is OK to use compat_futex_async() on a futex address on which
 * futex() WAKE operations are also performed.
 */

int lttng_ust_compat_futex_async(int32_t *uaddr, int op, int32_t val,
	const struct timespec *timeout, int32_t *uaddr2, int32_t val3)
{
	int ret = 0;

	/*
	 * Check if NULL. Don't let users expect that they are taken into
	 * account.
	 */
	assert(!timeout);
	assert(!uaddr2);
	assert(!val3);

	/*
	 * Ensure previous memory operations on uaddr have completed.
	 */
	cmm_smp_mb();

	switch (op) {
	case FUTEX_WAIT:
		while (CMM_LOAD_SHARED(*uaddr) == val) {
			if (poll(NULL, 0, 10) < 0) {
				ret = -1;
				/* Keep poll errno. Caller handles EINTR. */
				goto end;
			}
		}
		break;
	case FUTEX_WAKE:
		break;
	default:
		errno = EINVAL;
		ret = -1;
	}
end:
	return ret;
}
